///////////////////////////////////////////////////////////////////////////////
// HEIG-VD, Haute Ecole d'Ingenierie et de Gestion du canton de Vaud
// Institut REDS, Reconfigurable & Embedded Digital Systems
//
// File         : M2M_checker.sv
//
// Description  :
//
// Author       : S. Masle
// Date         : 19.10.2009
// Version      : 0.0
//
// Use          : testbench for Math2Mat
//
//| Modifications |////////////////////////////////////////////////////////////
// Version   Author Date               Description
//  01.00     guo   20100419            add OVM reporting
//                                      objection bugfix in service_M2M_Check_Output
//  01.01     guo   20100429            add Reporting
//  01.02     guo   20100503            add Timing Checker
//  01.03     guo   20100816            add Internal Checker
///////////////////////////////////////////////////////////////////////////////

`ifndef M2M_CHECKER__SV
  `define M2M_CHECKER__SV

`include "bases.sv"
`include "objection.sv"
`include "M2M_input_classes.sv"
`include "M2M_output_classes.sv"
`include "M2M_internal_classes.sv"

//_____________________________________________ class M2M_Checker ______________
/**
* Checker class to confirm proper operation of the Math2Mat Operation
* design. Result transactions from the DUT's output are received on
* channel "sink".
*
* Expected transactions are constructed from the output files made by
* Octave. service_M2M_Input() and predict_DUT_output(int fd, int output_nbr)
* are responsible to create those expected transactions.
*
* Expected transactions are are placed on the queue of expected results.
* There is one queue for each output channel. Output transactions are
* compared with the next item from the queue of expected results each time
* a new output transaction occurs.
*/
class M2M_Checker extends Component;


//-- 20100409 v01.00 -- guo -- comment -- error counting obsolete because already implemented in ovm_report
//  /**
//  * integer to save the number of errors
//  */
//  int Nb_Error;
//  int Prev_Nb_Error;

  /**
  * integer to know how much we have to shift when some outputs are parallel
  *
  * ex: outputs : a - b[1] - b[2] - b[3] - c
  *
  * index of c = 5 => c is the 3rd output => index = 3 + offset where offset = 2 (size of b - 1)
  */
  int offset = 0;

  /**
  * Array of channel - this checker will receive transactions
  * from the output monitor component over this array of channels.
  * In this array, there is one entry for each DUT output.
  */
  Channel sink[1:`NUMBER_OF_DUT_OUTPUTS]; // Result transactions from DUT output

  /**
  * Array of queues of M2M_Out_Trans that contains results expected
  * to be seen at the DUT's output.
  */
  protected M2M_Out_Trans expected[1:`NUMBER_OF_DUT_OUTPUTS] [$];

  // Whenever this class has some data on its expected-results
  // queue, the test needs to continue.  When there is no data
  // on the expected-results queue, this checker can be stopped
  // if other parts of the testbench decide the test is finished.
  //
  protected Objection objection_to_stop;

  // Internal data and methods that this checker needs to do its work.
  // The methods are "protected virtual" so that an extension to this
  // class could provide modified versions, if necessary.

  extern protected virtual task service_M2M_Input  ();
  extern protected virtual task service_M2M_Launch_Check ();
  extern protected virtual task service_M2M_Check_Output(input integer output_nbr);

  // predict_DUT_output() is a model of the DUT that computes the
  // expected output and places it on the queue of expected results
  //
  extern protected virtual function void predict_DUT_output (int fd, int output_nbr);

  int max_bit_error;
  
  // Overrides for the standard Component methods new(), body()
  //
  function new(string _name, Component _parent);
    super.new(_name, _parent);
    objection_to_stop = new(this);
  endfunction : new

  /**
  * The body task launches two task that will work in parallel.
  * The body task will end when the two task will have finished their work.
  */
  task body();
    max_bit_error=0;
    fork
      service_M2M_Input();
      service_M2M_Launch_Check();
    join
  endtask : body

endclass : M2M_Checker


//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//                       IMPLEMENTATIONS                         //
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//


// _____________________________________________________ service_M2M_Input ___
/**
* This task open each expected result file (generated by Octave) and launches
* the predict_DUT_output() task. A predict_DUT_output task is launched for each
* file.
*/
task M2M_Checker::service_M2M_Input();

  int fd;  //file descriptor

  for (int i = 1; i <= `NUMBER_OF_OUTPUTS; i ++) begin
      string fileName;    //file name that will change each loop

      //separate each case to form the name of the file because "%d" replaces leading zeroes by spaces
      //we can have until 999 outputs without having troubles, which seems enough
      if (i < 10)
          $sformat(fileName, "iofiles/file_output%1d.dat", i);
      else if (i < 100)
          $sformat(fileName, "iofiles/file_output%2d.dat", i);
      else
          $sformat(fileName, "iofiles/file_output%3d.dat", i);

      //open the file corresponding to the output
      fd = $fopen(fileName, "rb");

      //... and get the expected outputs from the file
      predict_DUT_output(fd, i);
  end // for

endtask


// _____________________________________________________ service_M2M_Launch_Check ___
/**
* This task launches service_M2M_Check_Output() as many times as there is outputs on the DUT
*/
task M2M_Checker::service_M2M_Launch_Check();

  //launch as many service_M2M_Check_Output task as we have outputs
  for (integer i = 1; i <= `NUMBER_OF_DUT_OUTPUTS; i++ ) begin
    fork
      automatic int j = i;
      service_M2M_Check_Output(j);
    join_none
  end

endtask

// _____________________________________________________ service_M2M_Check_Output ___
/**
* Each time a transaction is received from the output monitor, this transaction is
* compared with the first transaction in the queue.
*/
task M2M_Checker::service_M2M_Check_Output(input integer output_nbr);

  Transaction   from_DUT;
  M2M_Out_Trans exp;

  int sampleNb=0;

  int biterror;
//-- 20100419 v01.00 -- guo -- add -- reporting information
  string msg;
  
  
  `M2M_REPORT_VARIABLE("StartSim",0)
  objection_to_stop.raise();

  forever begin

    // Wait until the DUT produces some data.
    sink[output_nbr].get(from_DUT);

//-- 20100419 v01.00 -- guo -- change -- bugfix: never happens, as TB finisheds before, should be set to NUMBER_OF_STIMULI
//    if (expected[output_nbr].size() == 1)
//      objection_to_stop.raise();

    // Before trying to pop anything off the expected
    // results queue, check that it's not empty!
    assert (expected[output_nbr].size() > 0)
    else begin
      `M2M_REPORT_ERROR({"Got unexpected ", from_DUT.psprint(), "."})

      continue;
    end

    // Get the next expected result.
    exp = expected[output_nbr].pop_front();
    $sformat(
      msg,
      "DUT output no. %0d.%0d: %s",
      output_nbr, expected[output_nbr].size(), from_DUT.psprint(),
    );


    biterror=exp.bitdiff(from_DUT);
    if (biterror>max_bit_error)
      max_bit_error=biterror;

    // Check that expected matches actual output
    assert ( exp.compare(from_DUT) ) begin: pass_block
      `M2M_REPORT_INFO({ "Correct ", msg } )

    end: pass_block
    else begin: fail_block
      // Report the error
      $sformat(
        msg,
          {"Bad ", msg, " expected: %s"},
        exp.psprint()
      );
      `M2M_REPORT_ERROR(msg)
      

//-- 20100409 v01.00 -- guo -- comment -- error counting obsolete because already implemented in ovm_report
//      Nb_Error++;
    end: fail_block
//-- 20100409 v01.00 -- guo -- comment -- error counting obsolete because already implemented in ovm_report
//    if (Nb_Error != Prev_Nb_Error)
//      $display("Number of errors : %d", Nb_Error);
//
//    Prev_Nb_Error = Nb_Error;

    // If we just emptied the expected-results queue, then this
    // checker is no longer busy and it is OK for simulation to stop
    if (expected[output_nbr].size() == 0) begin

//-- 20100409 v01.00 -- guo -- comment -- error counting obsolete because already implemented in ovm_report
//     $display("********** Number of errors : %d", Nb_Error);

      `M2M_REPORT_INFO("Expected results queue emptied.")

//-- 20100419 v01.00 -- guo -- change -- quit forever loop
//      objection_to_stop.drop();
      break;
    end

   sampleNb++;
   if (output_nbr==1)
       if ((sampleNb % 50)==0)

        `M2M_REPORT_VARIABLE("Progress",50)

  end // forever
  
  
  `M2M_REPORT_VARIABLE("MaxBitError",max_bit_error)

//-- 20100419 v01.00 -- guo -- add -- droping objection
  objection_to_stop.drop();

endtask


// ______________________________________________ predict_DUT_output ___
/**
* This function distribute the samples in the file into the channels. If the DUT's output is parallel,
* the function loops to distribute one sample in each channel at each iteration of the loop.
* In the case the output is parallel, the function also increases an offset to keep in memory the number
* of filled parallel queues.
* If the case the output is serial, the function send all data in the same queue.
*
* @param int fd file descriptor of the expected result file
* @param int output_nbr the number of the output corresponding to the file opened (one file per output)
*/
function void M2M_Checker::predict_DUT_output(int fd, int output_nbr);

  M2M_Out_Trans predict_Trans;

  logic [31:0]  samples_Number; //number of samples in the file
  logic [31:0]  vector_Length;  //length of the vector of each sample

  logic [`M2M_BUS_SIZE-1:0] result;

  predict_Trans = new();

  void'($fread(samples_Number, fd));
  void'($fread(vector_Length, fd));

  //if outputs are parallel, read the file and send each data in a different queue
  if (OUTPUT_PARALLEL[output_nbr] == 1'b1) begin
      repeat (samples_Number) begin
          for (int i = 0; i < vector_Length; i++) begin
              //get the next result value in the file
              void'($fread(result, fd));

              predict_Trans.Result = result;

              //put the expected result in the right queue
              expected[offset+output_nbr+i].push_back(predict_Trans.m2m_copy());
          end
      end
      offset += vector_Length - 1;
  end
  //else send all data in the same queue
  else begin
      repeat (samples_Number*vector_Length) begin
          //get the next result value in the file
          void'($fread(result, fd));

          predict_Trans.Result = result;
          //put the expected result in the queue
          expected[offset+output_nbr].push_back(predict_Trans.m2m_copy());
      end
  end

endfunction : predict_DUT_output


//-- 20100503 v01.02 -- guo -- add -- add Timing Checker and services
//_____________________________________________ class M2M_Timing_Checker _______
/**
 *  The timing checker collects statistics about input and output timings.
 */
class M2M_Timing_Checker extends Component;

  /**
  * Arrays of channels - this checker will receive transactions
  * from the output monitor component as well as the input monitor component
  * over this arrays of channels.
  * In this arrays, there is one entry for each DUT input and output.
  */
  Channel sink_inputs [1:`NUMBER_OF_DUT_INPUTS];
  Channel sink_outputs[1:`NUMBER_OF_DUT_OUTPUTS];

  /**
   * Whenever this class has some data on its inputs
   * queue, the test needs to continue.  When there is no data
   * on the inputs queue, this checker can be stopped
   * if other parts of the testbench decide the test is finished.
   */
  protected Objection objection_to_stop;

  /**
   * Internal data and methods that this checker needs to do its work.
   * The methods are "protected virtual" so that an extension to this
   * class could provide modified versions, if necessary.
   */
  extern protected virtual task service_M2M_Latency();
  extern protected virtual task service_M2M_CalculationTime();
  extern protected virtual task service_M2M_InputRate();// empties the sink_inputs channel


  // YTA
  /**
   * Semaphore for synchronization between the 3 services. It allows
   * the first two to peek a transaction before the get of the third
   * service.
   */
  semaphore sem_input;
   
  /**
   * Override function new()
   */
  function new(string _name, Component _parent);
    super.new(_name, _parent);
    objection_to_stop = new(this);
  endfunction : new

  /**
   * Override task body() and fork the services
   */
  task body();
    sem_input = new();
    fork
      service_M2M_Latency();
      service_M2M_CalculationTime();
      service_M2M_InputRate();
    join
  endtask : body

endclass : M2M_Timing_Checker

`endif

// _____________________________________________________ service_M2M_Latency ___
/**
* Checks the time from first DUT input to first DUT output
*/
task M2M_Timing_Checker::service_M2M_Latency();

  Transaction   to_DUT;
  Transaction   from_DUT;
  time          latency;

  string msg;
  objection_to_stop.raise();

  // Wait until the stimuli driver produces first data and get them without removing
  sink_inputs[1].peek(to_DUT);
  // Signals that there is no more need of sink_input
  sem_input.put(1);
  
  latency = $time;
  // Wait until the DUT produces first data and get them without removing
  sink_outputs[1].peek(from_DUT);
  // calculate latency time
  latency = $time - latency;
  // calculate latency in clock cycles
  latency /= `CLOCK_TICKS;
  // reporting
  $sformat(
    msg,
    "Latency (time from first input to first output): %0d clock cycle(s)",
    latency,
  );
  `M2M_REPORT_INFO( msg )
  
  `M2M_REPORT_VARIABLE("Latency",latency)
//  $sformat(msg,"M2MVar(Latency)=%0d",latency);
//  $display(msg);

  objection_to_stop.drop();

endtask

// _____________________________________________ service_M2M_CalculationTime ___
/**
 * Checks the time from first DUT input to last DUT output
 */
task M2M_Timing_Checker::service_M2M_CalculationTime();

  Transaction   to_DUT;         //DUT input transaction
  Transaction   from_DUT;       //DUT output transaction
  int fd;                       //file descriptor
  int r;                        //status
  logic [31:0]  samples_Number; //number of samples
  time          period;         //total time from first to last input

  string msg;
  objection_to_stop.raise();

  // get number of samples from input file
  fd  = $fopen("iofiles/file_input1.dat", "rb"); //open the file corresponding to the input
  r   = $fread(samples_Number, fd);           // get number of samples from file
  $fclose(fd);

  // Wait until the stimuli driver produces first data and get them without removing
  sink_inputs[1].peek(to_DUT);
  
  // Signals that there is no more need of sink_input
  sem_input.put(1);
  
  period = $time;
  // Wait until the DUT produces last data and get them without removing
  repeat (samples_Number) begin
    sink_outputs[1].peek(from_DUT);
    sink_outputs[1].await_empty();
  end
  // calculate calculation time
  period = $time - period;
  // calculate calculation time in clock cycles
  period /= `CLOCK_TICKS;
  // reporting
  $sformat(
    msg,
    "Total Calculation Time (time from first input to last output): %0d clock cycles for %0d inputs",
    period, samples_Number
  );
  `M2M_REPORT_INFO( msg )
  
  // Simple message that is 
  $sformat(msg,"M2MVar(TotalTime)=%0d",period);
  $display(msg);
  $sformat(msg,"M2MVar(NbInputs)=%0d",samples_Number);
  $display(msg);

  objection_to_stop.drop();

endtask

// ___________________________________________________ service_M2M_InputRate ___
/**
* Checks the time from first DUT input to last DUT input
*/
task M2M_Timing_Checker::service_M2M_InputRate();

  Transaction   to_DUT;         //transaction
  int fd;                       //file descriptor
  int r;                        //status
  logic [31:0]  samples_Number; //number of samples
  time          period;         //total time from first to last input
  time          rate;           //average input time between two inputs

  string msg;
  objection_to_stop.raise();

  // get number of samples from input file
  fd  = $fopen("iofiles/file_input1.dat", "rb"); //open the file corresponding to the input
  r   = $fread(samples_Number, fd);           // get number of samples from file
  $fclose(fd);

 
  // Waits for the 2 other services that need to peek on sink_inputs
  sem_input.get(2);
  
  // wait until first data is put into DUT
  sink_inputs[1].get(to_DUT);
  period = $time;
//  $display("-- GUO -- 501 -- period: %0d", period);
  // wait until last data is put into DUT
  // calculate overall input time
  repeat (samples_Number-1) begin
    sink_inputs[1].get(to_DUT);
  end
//  while ( !sink_inputs[1].empty() @ );
//  end
//  sink_inputs[1].await_empty();
  period = $time - period;
//  $display("-- GUO -- 511 -- period: %0d", period);
  // calculate input time in clock cycles
  period /= `CLOCK_TICKS;
  // calculate input rate
  rate = period / (samples_Number-1);
  // reporting
  $sformat(
    msg,
    "Total Input Time (time from first input to last input): %0d clock cycles for %0d inputs",
    period, samples_Number,
  );
  `M2M_REPORT_INFO( msg )
  $sformat(
    msg,
    "Input Rate (average time between two data inputs): %0d clock cycle(s)",
    rate,
  );
  `M2M_REPORT_INFO( msg )

  objection_to_stop.drop();

endtask

//-- 20100816 v01.03 -- guo -- add -- add Internal Checker and services
//_____________________________________________ class M2M_Checker ______________
/**
 * Checker class to confirm proper operation of the Math2Mat Operation
 * design. Result transactions from the DUT's internal signals are received on
 * channel "sink".
 *
 * Expected transactions are constructed from files made by Octave.
 * service_M2M_Internals() and predict_DUT_internals(int fd, int internals_nbr)
 * are responsible to create those expected transactions.
 *
 * Expected transactions are are placed on the queue of expected data.
 * There is one queue for each internal channel. Internal transactions are
 * compared with the next item from the queue of expected data each time
 * a new internal transaction occurs.
 *
 * This class has been created in a first approach to verify the functionality
 * of the internals monitor. Conceptualy it would be better to create a
 * base checker class and derive the outputs checker aswell as the internals
 * checker from it.
 */
class M2M_Internals_Checker extends Component;
  /**
   * integer to know how much we have to shift when some internals are parallel
   *
   * ex: internals : a - b[1] - b[2] - b[3] - c
   *
   * index of c = 5 => c is the 3rd internal => index = 3 + offset where offset = 2 (size of b - 1)
   */
  int offset = 0;

  /**
   * Array of channels - this checker receives transactions from the internals
   * monitor component over this array of channels.
   * In this array, there is one entry for each DUT internal to monitor.
   */
  Channel sink[1:`NUMBER_OF_DUT_INTERNALS];

  /**
   * Array of queues of M2M_Internal_Trans that contains results expected
   * to be seen on the DUT's internals.
   */
  protected M2M_Internal_Trans expected[1:`NUMBER_OF_DUT_INTERNALS] [$];


  /**
   * Whenever this class has some data on its expected-data queue, the test
   * needs to continue. When there is no more data on the expected-data queue,
   * this checker can be stopped if all other parts of the testbench also
   * decide the test is finished.
   */
  protected Objection objection_to_stop;

  /**
   * Internal data and methods that this checker needs to do its work.
   * The methods are "protected virtual" so that an extension to this class
   * could provide modified versions, if necessary.
   */
  extern protected virtual task service_M2M_Internals();
  extern protected virtual task service_M2M_Launch_Check_Internals();
  extern protected virtual task service_M2M_Check_Internals(
                                                   input integer internals_nbr);

  /**
   * predict_DUT_internals() gets the expected results from a expected result
   * file (generated by Octave) and puts them on the queue of expected results
   */
  extern protected virtual function void predict_DUT_internals(
                                                             int fd,
                                                             int internals_nbr);

  /**
   * Overrides for the standard Component methods new(), body()
   */
  function new(string _name, Component _parent);
    super.new(_name, _parent);
    objection_to_stop = new(this);
  endfunction : new

  /**
   * The body task launches two task that work in parallel.
   * The body task will end when the two task have finished their work.
   */
  task body();
    fork
      service_M2M_Internals();
      service_M2M_Launch_Check_Internals();
    join
  endtask : body

endclass : M2M_Internals_Checker

// ___________________________________________________ service_M2M_Internals ___
/**
 * This task open each expected result file (generated by Octave) and launches
 * the predict_DUT_output() task. A predict_DUT_output task is launched for each
 * file.
 */
task M2M_Internals_Checker::service_M2M_Internals();

  int fd;  //file descriptor
  string msg;

  for (int i = 1; i <= `NUMBER_OF_DUT_INTERNALS; i ++) begin
    string fileName;    //file name that will change each loop

    /**
     * separate each case to form the name of the file because "%d" replaces
     * leading zeroes by spaces we can have until 999 outputs without having
     * troubles, which seems enough
     */
    if (i < 10)
      $sformat(fileName, "iofiles/file_internal%1d.dat", i);
    else if (i < 100)
      $sformat(fileName, "iofiles/file_internal%2d.dat", i);
    else
      $sformat(fileName, "iofiles/file_internal%3d.dat", i);

    /**
     * open the file corresponding to the output
     */
    fd = $fopen(fileName, "rb");
    assert (fd)
    else begin
      $sformat(
        msg,
        {"Error opening file %s"},
        fileName
      );
      `M2M_REPORT_ERROR(msg)
    end //: else

    /**
     * ... and get the expected outputs from the file
     */
    predict_DUT_internals(fd, i);

    $fclose(fd);

  end //: for

endtask


// ______________________________________ service_M2M_Launch_Check_Internals ___
/**
 * This task launches service_M2M_Check_Internals() as many times as there are
 * internals of the DUT to be monitored
 */
task M2M_Internals_Checker::service_M2M_Launch_Check_Internals();

  //launch as many service_M2M_Check_Internals task as we have outputs
  for (integer i = 1; i <= `NUMBER_OF_DUT_INTERNALS; i++ ) begin
    fork
      automatic int j = i;
      service_M2M_Check_Internals(j);
    join_none
  end

endtask

// _____________________________________________ service_M2M_Check_Internals ___
/**
 * Each time a transaction is received from the internal monitor, this
 * transaction is compared with the first transaction in the queue.
 */
task M2M_Internals_Checker::service_M2M_Check_Internals(
                                                 input integer internals_nbr);

  Transaction   from_DUT;
  M2M_Internal_Trans exp;

  string msg;
  objection_to_stop.raise();

  forever begin

    /**
     * Wait until the DUT produces some data.
     */
    sink[internals_nbr].get(from_DUT);

    /**
     * Before trying to pop anything off the expected data queue, check that
     * it's not empty!
     */
    assert (expected[internals_nbr].size() > 0)
    else begin

      `M2M_REPORT_ERROR({"Got unexpected ", from_DUT.psprint(), "."})
      continue; // skip to end of forever loop

    end //: assert

    /**
     * Get the next expected data
     */
    exp = expected[internals_nbr].pop_front();
    $sformat(
      msg,
      "DUT internal no. %0d.%0d: %s",
      internals_nbr, expected[internals_nbr].size(), from_DUT.psprint(),
    );

    // Check that expected matches actual output
    assert ( exp.compare(from_DUT) ) begin: pass_block

      `M2M_REPORT_INFO( { "Correct ", msg } )

    end: pass_block
    else begin: fail_block

      $sformat(
        msg,
        {"Bad ", msg, " expected: %s"},
        exp.psprint()
      );
      `M2M_REPORT_ERROR(msg)

    end: fail_block

    /**
     * If we just emptied the expected-results queue, then this checker is no
     * longer busy and it is OK for simulation to stop
     */
    if (expected[internals_nbr].size() == 0) begin

      `M2M_REPORT_INFO("Expected results queue emptied.")

      break; // out of forever loop

    end //: if

  end // forever

  objection_to_stop.drop();

endtask


// ___________________________________________________ predict_DUT_internals ___
/**
 * This function distribute the samples in the file into the channels. If the
 * DUT's output is parallel, the function loops to distribute one sample in
 * each channel at each iteration of the loop. In the case the output is
 * parallel, the function also increases an offset to keep in memory the number
 * of filled parallel queues.
 * If the case the output is serial, the function send all data in the same
 * queue.
 *
 * @param int fd file descriptor of the expected result file
 * @param int internals_nbr the number of the output corresponding to the file
 *                          opened (one file per output)
 */
function void M2M_Internals_Checker::predict_DUT_internals(
                                                            int fd,
                                                            int internals_nbr);

  int code;
  string msg;
  M2M_Internal_Trans predict_Trans;

  /**
   * number of samples in the file
   */
  logic [31:0]  samples_Number;
  /**
   * length of the vector of each sample
   */
  logic [31:0]  vector_Length;

  logic [`M2M_BUS_SIZE-1:0] result;

  predict_Trans = new();

  assert ($fread(samples_Number, fd))
  else begin
    $sformat(
      msg,
      {"Error reading file %s"},
      fd
    );
    `M2M_REPORT_ERROR(msg)
  end //: else

  assert ($fread(vector_Length, fd))
  else begin
    $sformat(
      msg,
      {"Error reading file %s"},
      fd
    );
    `M2M_REPORT_ERROR(msg)
  end //: else

  /**
   * if internals are parallel, read the file and send each data in a different
   * queue
   */
  if (INTERNAL_PARALLEL[internals_nbr] == 1'b1) begin

    repeat (samples_Number) begin

      for (int i = 0; i < vector_Length; i++) begin

        /**
         * get the next result value in the file
         */
        assert ($fread(result, fd))
        else begin
          $sformat(
            msg,
            {"Error reading file %s"},
            fd
          );
          `M2M_REPORT_ERROR(msg)
        end //: else

        predict_Trans.Data = result;

        /**
         * put the expected result in the right queue
         */
        expected[offset+internals_nbr+i].push_back(predict_Trans.m2m_copy());

      end //: for

    end //: repeat

    offset += vector_Length - 1;

  end //: if
  /**
   * else send all data in the same queue
   */
  else begin

    repeat (samples_Number*vector_Length) begin

      /**
       * get the next result value in the file
       */
      assert ($fread(result, fd))
      else begin
        $sformat(
          msg,
          {"Error reading file %s"},
          fd
        );
        `M2M_REPORT_ERROR(msg)
      end //: else

      predict_Trans.Data = result;

      /**
       * put the expected result in the queue
       */
      expected[offset+internals_nbr].push_back(predict_Trans.m2m_copy());

    end //: repeat

  end //: else

endfunction : predict_DUT_internals

`endif
